/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 2013 Rohde & Schwarz
** MODULE:       CSCPIFileAccess_PR100.cpp
** LANGUAGE:     C/C++
** AUTHOR:       
** ABSTRACT:     Example to access files in a PR100
** PREMISES:
** REMARKS:
** HISTORY:
********************************************************************************/

/* INCLUDE FILES ***************************************************************/
#include "CSCPIFileAccess_PR100.h"

/* CFileAccess Implementation ***************************************************************/
CSCPIFileAccess::CSCPIFileAccess(CSCPIHelper& device, CSCPIHelper::CallbackFunc callback, void * pPrivate) :
    m_device           (device),
    m_callBackFunc     (callback),
    m_pCallbackPrivate (pPrivate),
    m_bDebugOutput     (false)
{

}

int CSCPIFileAccess::listfiles( const std::string& path, std::vector<CSCPIFileAccess::FILE_PROPERTY>& result )
{
    FILE_PROPERTY prop;
    char cBuffer[READ_BUFFER_SIZE_BYTES];
    int len = 0;
    int nResponseErr = 0;
    int nSCPIErr = 0;

    std::string cmd;

    //Change the directory to the device output directory
    cmd = "mmemory:cdirectory '";
    cmd.append(path);
    cmd.append("'\n");
    m_device.SendCmd(cmd);

    nSCPIErr = m_device.GetLastCmdErr();
    if(nSCPIErr != 0)
    {
        if(m_bDebugOutput) printf("ERROR: mmemory:cdirectory err = %d\n", nSCPIErr);
        goto err_cleanup;
    }

    //Get the list of files
    cmd = "MMEMory:CATalog?\n";
    m_device.SendCmd(cmd);

    len = sizeof(cBuffer);
    nResponseErr = m_device.GetResponseWithTimeout(cBuffer, len, READ_TIMEOUT_MS);
    if(nResponseErr != 0)
    {
        nSCPIErr = m_device.GetLastCmdErr();

        //If SCPI has no error, means that the error is not SCPI related,
        //but maybe the transfer broken or unreliable. In that case return
        //the nErr value
        if(nSCPIErr == 0) nSCPIErr = nResponseErr;

        if(m_bDebugOutput) printf("ERROR: mmemory:catalog? SCPI_ERR = %d\n", nSCPIErr);
        goto err_cleanup;
    }

    // Parse the returned result
    char * token      = NULL;
    char * next_token = NULL;
    int field         = -2;

    token = CSCPIHelper::Tokenise(cBuffer,",", &next_token);
    
    while (token != NULL)
    {
        switch(field)
        {
            case -2: //used storage
                if(m_bDebugOutput) printf("%s bytes used\n", token);
                break;

            case -1: //available storage
                if(m_bDebugOutput) printf("%s bytes available\n", token);
                break;

            case 0: //file_name
                prop.name = token;
                break;
            case 1: //file_extension
                prop.ext = token;
                break;
            case 2: //file_size
                prop.size = atoi(token);
                break;
            case 3: //file_date yyyy
                prop.date = token;
                break;
            case 4: //file_date mm
                prop.date.append("/");
                prop.date.append(token);
                break;
            case 5: //file_date dd
                prop.date.append("/");
                prop.date.append(token);
                break;
            case 6: //file_time hh
                prop.time = token;
                break;
            case 7: //file_date mm
                prop.time.append(":");
                prop.time.append(token);
                break;
            case 8: //file_date ss
                prop.time.append(":");
                prop.time.append(token);

                // End of one file, populate structure with other info and 
                // add structure to vector
                prop.path       = path;
                prop.devAddress = m_device.DeviceAddress();
                prop.devName    = m_device.DeviceName();
                prop.devSN      = m_device.DeviceSN();
                result.push_back(prop);
                break;

            default:
                // Should never reach here
                break;
        }

        field++;
        if(field >= 9) field = field % 9;
        token = CSCPIHelper::Tokenise (NULL, ",", &next_token);
    }

    return nSCPIErr;

err_cleanup:
    return nSCPIErr;
}


int CSCPIFileAccess::download( const std::string& remotefile )
{
    bool bErr = false;
    int  nSCPIErr = 0;
    std::string cmd;

    //Send command to get the data
    cmd = "mmemory:data? '";
    cmd.append(remotefile);
    cmd.append("'\n");
    m_device.SendCmd(cmd);

    //Process the block data response
    int nErr = m_device.GetBlockData(m_callBackFunc, m_pCallbackPrivate);
    if(nErr != 0)
    {
        nSCPIErr = m_device.GetLastCmdErr();

        //If SCPI has no error, means that the error is not SCPI related,
        //but maybe the transfer broken or unreliable. In that case return
        //the nErr value
        if(nSCPIErr == 0) nSCPIErr = nErr; 

        if(m_bDebugOutput) printf("ERROR: mmemory:data? err=%d\n", nSCPIErr);
    }

    return nSCPIErr;

}


